<?php
// Enable error reporting for debugging (remove in production)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set content type
header('Content-Type: application/json');

// Allow CORS if needed
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Check if form was submitted via POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Validate and sanitize input data
function sanitize_input($data) {
    return htmlspecialchars(strip_tags(trim($data)));
}

// Get form data
$name = sanitize_input($_POST['contactName'] ?? '');
$email = filter_var($_POST['contactEmail'] ?? '', FILTER_SANITIZE_EMAIL);
$subject = sanitize_input($_POST['contactSubject'] ?? '');
$message = sanitize_input($_POST['contactMessage'] ?? '');

// Validate required fields
$errors = [];
if (empty($name)) $errors[] = 'Name is required';
if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) $errors[] = 'Valid email is required';
if (empty($message)) $errors[] = 'Message is required';

// If there are validation errors, return them
if (!empty($errors)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => implode(', ', $errors)]);
    exit;
}

// Set default subject if empty
if (empty($subject)) {
    $subject = 'General Inquiry';
}

// Prepare email content
$to = 'info@londonhotel.co.tz';
$email_subject = 'Contact Form: ' . $subject . ' - London Hotel';
$from = 'contactform@londonhotel.co.tz';

// Create HTML email content
$email_message = "
<!DOCTYPE html>
<html>
<head>
    <meta charset='UTF-8'>
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
        .container { max-width: 600px; margin: 0 auto; padding: 20px; }
        .header { background-color: #ffc107; color: #000; padding: 20px; text-align: center; }
        .content { background-color: #f9f9f9; padding: 20px; }
        .contact-details { background-color: #fff; padding: 15px; margin: 10px 0; border-left: 4px solid #ffc107; }
        .message-content { background-color: #fff; padding: 20px; margin: 15px 0; border-radius: 5px; }
        .footer { background-color: #333; color: #fff; padding: 15px; text-align: center; }
        .label { font-weight: bold; color: #555; }
        .value { margin-bottom: 10px; }
    </style>
</head>
<body>
    <div class='container'>
        <div class='header'>
            <h1>📧 New Contact Message</h1>
            <p>London Hotel - Dar es Salaam</p>
        </div>
        
        <div class='content'>
            <h2>Contact Information</h2>
            <div class='contact-details'>
                <div class='value'><span class='label'>Name:</span> {$name}</div>
                <div class='value'><span class='label'>Email:</span> {$email}</div>
                <div class='value'><span class='label'>Subject:</span> {$subject}</div>
            </div>
            
            <h2>Message</h2>
            <div class='message-content'>
                <p>" . nl2br($message) . "</p>
            </div>
            
            <p><strong>Next Steps:</strong></p>
            <ul>
                <li>Reply to the customer within 24 hours</li>
                <li>Address their inquiry or concern</li>
                <li>Provide any requested information</li>
            </ul>
        </div>
        
        <div class='footer'>
            <p>London Hotel | Ubungo Maziwa, Dar es Salaam | +255 658 855268</p>
            <p>This message was submitted on " . date('Y-m-d H:i:s') . "</p>
        </div>
    </div>
</body>
</html>";

// Email headers
$headers = [
    'From: ' . $from,
    'Reply-To: ' . $email,
    'MIME-Version: 1.0',
    'Content-Type: text/html; charset=UTF-8',
    'X-Mailer: PHP/' . phpversion()
];

// Send email
$mail_sent = mail($to, $email_subject, $email_message, implode("\r\n", $headers));

if ($mail_sent) {
    // Log successful contact (optional)
    $log_entry = date('Y-m-d H:i:s') . " - Contact message from {$name} ({$email}) - Subject: {$subject}\n";
    file_put_contents('contact_logs.txt', $log_entry, FILE_APPEND | LOCK_EX);
    
    echo json_encode([
        'success' => true, 
        'message' => 'Thank you for your message! We will get back to you within 24 hours.'
    ]);
} else {
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'message' => 'Sorry, there was an error sending your message. Please try again or contact us directly at +255 658 855268.'
    ]);
}
?>

